# -*- coding: utf-8 -*-
"""
:mod:`EdgarRenderer.EdgarRenderer`
~~~~~~~~~~~~~~~~~~~
Edgar(tm) Renderer was created by staff of the U.S. Securities and Exchange Commission.
Data and content created by government employees within the scope of their employment 
are not subject to domestic copyright protection. 17 U.S.C. 105.
"""
VERSION="3.3.0.814"

from collections import defaultdict
from arelle import PythonUtil  # define 2.x or 3.x string types
PythonUtil.noop(0)  # Get rid of warning on PythonUtil import
from arelle import (Cntlr, FileSource, ModelDocument, XmlUtil, Version, ModelValue, Locale, PluginManager, WebCache, ModelFormulaObject,
                    ViewFileFactList, ViewFileFactTable, ViewFileConcepts, ViewFileFormulae,
                    ViewFileRelationshipSet, ViewFileTests, ViewFileRssFeed, ViewFileRoleTypes)
import RefManager, IoManager, Inline, Utils, Filing, Summary
import datetime, zipfile, logging, shutil, gettext, time, shlex, sys, traceback, linecache, os
from lxml import etree
from os import getcwd, remove, removedirs
from os.path import join, isfile, exists, dirname, basename, isdir
from optparse import OptionParser, SUPPRESS_HELP


def main():
    """Main program to initiate application from command line or as a separate process (e.g, java Runtime.getRuntime().exec).  May perform
    a command line request, or initiate a web server on specified local port.       
       :param argv: Command line arguments.  (Currently supported arguments can be displayed by the parameter *--help*.)
       :type message: [str]
       """
    envArgs = os.getenv("ARELLE_ARGS")
    if envArgs:
        args = shlex.split(envArgs)
    else:
        args = sys.argv[1:]        
    gettext.install("arelle")  # needed for options messages      

    controller = EdgarRenderer()  # need controller for plug ins to be loaded
    (options, rendererOk) = parseOptions(controller, args)
    if rendererOk:
        controller.runRenderer(options)
        
# Helper functions

# def linenum():
    # Returns the current line number in the .py file
    # return inspect.currentframe().f_back.f_lineno


###############

def parseOptions(controller, args):
    """interface used by Main program
    """
    # controller = CntlrCmdLine()  # need controller for plug ins to be loaded
    usage = "Usage: %prog [options]"
    
    parser = OptionParser(usage,
                          version="Arelle(r) {0}bit {1}".format(controller.systemWordSize, Version.version),
                          conflict_handler="resolve")  # allow reloading plug-in options without errors
    # parser options that can be overridden in the config file do not have a default specified here.
    parser.add_option("-f", "--file", dest="entrypoint", default='.',
                      help=_("LOCATION is an entry point, which may be "
                             "an XBRL instance, schema, linkbase file, "
                             "a zip file that contains the DTS (instance/schema/linkbase files), "
                             "inline XBRL instance, testcase file, "
                             "testcase index file.  FILENAME may be "
                             "a local file, a URI to a web located file, "
                             "or a folder that contains the DTS (instance/schema/linkbase files), "
                             ))   
    parser.add_option("-o", "--output", dest="zipOutputFile",
                      help=_("Zip the artifacts generated by the rendering process into a file with this name."))
    parser.add_option("-c", "--configFile", dest="configFile", default='EdgarRenderer.xml',
                      help=_("Path of location of Edgar Renderer configuration file relative to CWD. Default is EdgarRenderer.xml."))
    parser.add_option("-r", "--reports", dest="reportsFolder",
                     help=_("Relative path and name of the reports folder."))
    parser.add_option("--resources", dest="resourcesFolder",
                     help=_("Relative path and name of the resources folder, which includes static xslt, css and js support files."))
    parser.add_option("--filings", dest="filingsFolder",
                     help=_("Relative path and name of the filings (input) folder."))
    parser.add_option("--processing", dest="processingFolder",
                     help=_("Relative path and name of the processing folder."))
    parser.add_option("--errors", dest="errorsFolder",
                     help=_("Relative path and name of the errors folder, where unprocessable filings are stored."))
    parser.add_option("--delivery", dest="deliveryFolder",
                     help=_("Relative path and name of the delivery folder, where all rendered artifacts are stored."))
    parser.add_option("--archive", dest="archiveFolder",
                     help=_("Relative path and name of the archive folder, where successfully-processed original filings are stored."))
    parser.add_option("--processingfrequency", dest="processingFrequency",
                     help=_("The sleep time for the RE3 daemon shell if no XBRL filing is present in the staging area."))
    parser.add_option("--renderingService", dest="renderingService",
                     help=_("Type of service...Instance: one time rendering, or Daemon: background processing."))
    parser.add_option("--totalClean", dest="totalClean", action="store_true",
                     help=_("Boolean to indicate if the shell should completely clean the archive, errors and delivery folders before processing."))
    parser.add_option("--deleteProcessedFilings", dest="deleteProcessedFilings", action="store_true",
                     help=_("Boolean to indicate if processed filings should be deleted or not."))
    parser.add_option("--htmlReportFormat", dest="htmlReportFormat",
                     help=_("Type of HTML report...Complete: asPage rendering = True, or Fragment: asPage rendering = False."))

    parser.add_option("--reportFormat", dest="reportFormat",
                      help=_("One of Xml, Html, or HtmlAndXml."))
        
    parser.add_option("--failFile", dest="failFile", help=_("Relative path and name of fail file. "))
    
    parser.add_option("--reportXslt", dest="reportXslt", help=_("Path and name of Stylesheet for producing a report file."))
    parser.add_option("--summaryXslt", dest="summaryXslt", help=_("Path and name of Stylesheet, if any, for producing filing summary html."))
    parser.add_option("--excelXslt", dest="excelXslt", help=_("Path and name of Stylesheet, if any, for producing Excel 2007 xlsx output."))
    parser.add_option("--auxMetadata", action="store_true", dest="auxMetadata", help=_("Set flag to generate inline xbrl auxiliary files"))    
    Inline.saveTargetDocumentCommandLineOptionExtender(parser)
    parser.add_option("--sourceList", action="store", dest="sourceList", help=_("Comma-separated triples of instance file, doc type and source file."))
    parser.add_option("--copyInlineFilesToOutput", action="store_true", dest="copyInlineFilesToOutput", help=_("Set flag to copy all inline files to the output folder or zip."))
    parser.add_option("--noEquity", action="store_true", dest="noEquity", help=_("Set flag to suppress special treatment of Equity Statements. "))
            
    parser.add_option("--xdgConfigHome", action="store", dest="xdgConfigHome",
                      help=_("Specify non-standard location for configuration and cache files (overrides environment parameter XDG_CONFIG_HOME)."))
    parser.add_option("-v", "--validate",
                      action="store_true", dest="validate",
                      help=_("Validate the file according to the entry "
                             "file type.  If an XBRL file, it is validated "
                             "according to XBRL validation 2.1, calculation linkbase validation "
                             "if either --calcDecimals or --calcPrecision are specified, and "
                             "EDGAR Filing Manual (if --efm selected) or Global Filer Manual "
                             "disclosure system validation (if --gfm=XXX selected). "
                             "If a test suite or testcase, the test case variations "
                             "are individually so validated. "
                             "If formulae are present they will be validated and run unless --formula=none is specified. "
                             ))
    # not only does validation use the utr, so does unit rendering.  
    parser.add_option("--utr", action="store_true", dest="utrValidate",
                      help=_("Select validation with respect to Unit Type Registry irrespective of DTS taxonomy version."))
    parser.add_option("--utrUrl", action="store", dest="utrUrl",
                      help=_("Override disclosure systems Unit Type Registry location (URL or file path)."))
    parser.add_option("--utrurl", action="store", dest="utrUrl", help=SUPPRESS_HELP)
    
    # TODO - option --labelLang and labellang have no effect on rendering but someday it could.
    parser.add_option("--labelLang", action="store", dest="labelLang",
                      help=_("Language for labels in following file options (override system settings)"))
    parser.add_option("--labellang", action="store", dest="labelLang", help=SUPPRESS_HELP)
        
    # remaining options control details of arelle load and validate behavior that are not very relevant for rendering.
    parser.add_option("--username", dest="username",
                      help=_("user name if needed (with password) for web file retrieval"))
    parser.add_option("--password", dest="password",
                      help=_("password if needed (with user name) for web retrieval"))
    # special option for web interfaces to suppress closing an opened modelXbrl
    parser.add_option("--keepOpen", dest="keepOpen", action="store_true", help=SUPPRESS_HELP)
    parser.add_option("-i", "--import", dest="importFiles",
                      help=_("FILENAME is a list of files to import to the DTS, such as "
                             "additional formula or label linkbases.  "
                             "Multiple file names are separated by a '|' character. "))
    parser.add_option("-d", "--diff", dest="diffFile",
                      help=_("FILENAME is a second entry point when "
                             "comparing (diffing) two DTSes producing a versioning report."))
    parser.add_option("--calcDecimals", action="store_true", dest="calcDecimals",
                      help=_("Specify calculation linkbase validation inferring decimals."))
    parser.add_option("--calcdecimals", action="store_true", dest="calcDecimals", help=SUPPRESS_HELP)
    parser.add_option("--calcPrecision", action="store_true", dest="calcPrecision",
                      help=_("Specify calculation linkbase validation inferring precision."))
    parser.add_option("--calcprecision", action="store_true", dest="calcPrecision", help=SUPPRESS_HELP)
    parser.add_option("--efm", action="store_true", dest="validateEFM",
                      help=_("Select Edgar Filer Manual disclosure system validation (strict)."))
    parser.add_option("--gfm", action="store", dest="disclosureSystemName", help=SUPPRESS_HELP)
    parser.add_option("--disclosureSystem", action="store", dest="disclosureSystemName",
                      help=_("Specify a disclosure system name and"
                             " select disclosure system validation.  "
                             "Enter --disclosureSystem=help for list of names or help-verbose for list of names and descriptions. "))
    parser.add_option("--disclosuresystem", action="store", dest="disclosureSystemName", help=SUPPRESS_HELP)
    parser.add_option("--infoset", action="store_true", dest="infosetValidate",
                      help=_("Select validation with respect testcase infosets."))
    parser.add_option("--labelRole", action="store", dest="labelRole",
                      help=_("Label role for labels in following file options (instead of standard label)"))
    parser.add_option("--labelrole", action="store", dest="labelRole", help=SUPPRESS_HELP)
    parser.add_option("--facts", "--csvFacts", action="store", dest="factsFile",
                      help=_("Write fact list into FILE"))
    parser.add_option("--factListCols", action="store", dest="factListCols",
                      help=_("Columns for fact list file"))
    parser.add_option("--factTable", "--csvFactTable", action="store", dest="factTableFile",
                      help=_("Write fact table into FILE"))
    parser.add_option("--concepts", "--csvConcepts", action="store", dest="conceptsFile",
                      help=_("Write concepts into FILE"))
    parser.add_option("--pre", "--csvPre", action="store", dest="preFile",
                      help=_("Write presentation linkbase into FILE"))
    parser.add_option("--cal", "--csvCal", action="store", dest="calFile",
                      help=_("Write calculation linkbase into FILE"))
    parser.add_option("--dim", "--csvDim", action="store", dest="dimFile",
                      help=_("Write dimensions (of definition) linkbase into FILE"))
    parser.add_option("--formulae", "--htmlFormulae", action="store", dest="formulaeFile",
                      help=_("Write formulae linkbase into FILE"))
    parser.add_option("--viewArcrole", action="store", dest="viewArcrole",
                      help=_("Write linkbase relationships for viewArcrole into viewFile"))
    parser.add_option("--viewarcrole", action="store", dest="viewArcrole", help=SUPPRESS_HELP)
    parser.add_option("--viewFile", action="store", dest="viewFile",
                      help=_("Write linkbase relationships for viewArcrole into viewFile"))
    parser.add_option("--viewfile", action="store", dest="viewFile", help=SUPPRESS_HELP)
    parser.add_option("--roleTypes", action="store", dest="roleTypesFile",
                      help=_("Write defined role types into FILE"))
    parser.add_option("--roletypes", action="store", dest="roleTypesFile", help=SUPPRESS_HELP)
    parser.add_option("--arcroleTypes", action="store", dest="arcroleTypesFile",
                      help=_("Write defined arcrole types into FILE"))
    parser.add_option("--arcroletypes", action="store", dest="arcroleTypesFile", help=SUPPRESS_HELP)
    parser.add_option("--testReport", "--csvTestReport", action="store", dest="testReport",
                      help=_("Write test report of validation (of test cases) into FILE"))
    parser.add_option("--testreport", "--csvtestreport", action="store", dest="testReport", help=SUPPRESS_HELP)
    parser.add_option("--testReportCols", action="store", dest="testReportCols",
                      help=_("Columns for test report file"))
    parser.add_option("--testreportcols", action="store", dest="testReportCols", help=SUPPRESS_HELP)
    parser.add_option("--rssReport", action="store", dest="rssReport",
                      help=_("Write RSS report into FILE"))
    parser.add_option("--rssreport", action="store", dest="rssReport", help=SUPPRESS_HELP)
    parser.add_option("--rssReportCols", action="store", dest="rssReportCols",
                      help=_("Columns for RSS report file"))
    parser.add_option("--rssreportcols", action="store", dest="rssReportCols", help=SUPPRESS_HELP)
    parser.add_option("--logFile", action="store", dest="logFile",
                      help=_("Write log messages into file, otherwise they go to standard output.  " 
                             "If file ends in .xml it is xml-formatted, otherwise it is text. "))
    parser.add_option("--logfile", action="store", dest="logFile", help=SUPPRESS_HELP)
    parser.add_option("--logFormat", action="store", dest="logFormat",
                      help=_("Logging format for messages capture, otherwise default is \"[%(messageCode)s] %(message)s - %(file)s\"."))
    parser.add_option("--logformat", action="store", dest="logFormat", help=SUPPRESS_HELP)
    parser.add_option("--logLevel", action="store", dest="logLevel",
                      help=_("Minimum level for messages capture, otherwise the message is ignored.  " 
                             "Current order of levels are debug, info, info-semantic, warn, warn-semantic, warn, assertion-satisfied, inconsistency, error-semantic, assertion-not-satisfied, and error. "))
    parser.add_option("--loglevel", action="store", dest="logLevel", help=SUPPRESS_HELP)
    parser.add_option("--logLevelFilter", action="store", dest="logLevelFilter",
                      help=_("Regular expression filter for logLevel.  " 
                             "(E.g., to not match *-semantic levels, logLevelFilter=(?!^.*-semantic$)(.+). "))
    parser.add_option("--loglevelfilter", action="store", dest="logLevelFilter", help=SUPPRESS_HELP)
    parser.add_option("--logCodeFilter", action="store", dest="logCodeFilter",
                      help=_("Regular expression filter for log message code."))
    parser.add_option("--logcodefilter", action="store", dest="logCodeFilter", help=SUPPRESS_HELP)
    parser.add_option("--showOptions", action="store_true", dest="showOptions", help=SUPPRESS_HELP)
    parser.add_option("--showErrors", action="store_true", dest="showErrors",
                      help=_("List all errors and warnings that may occur during RE3 processing."))
    parser.add_option("--parameters", action="store", dest="parameters", help=_("Specify parameters for formula and validation (name=value[,name=value])."))
    parser.add_option("--parameterSeparator", action="store", dest="parameterSeparator", help=_("Specify parameters separator string (if other than comma)."))
    parser.add_option("--parameterseparator", action="store", dest="parameterSeparator", help=SUPPRESS_HELP)
    parser.add_option("--formula", choices=("validate", "run", "none"), dest="formulaAction",
                      help=_("Specify formula action: "
                             "validate - validate only, without running, "
                             "run - validate and run, or "
                             "none - prevent formula validation or running when also specifying -v or --validate.  "
                             "if this option is not specified, -v or --validate will validate and run formulas if present"))
    parser.add_option("--formulaParamExprResult", action="store_true", dest="formulaParamExprResult", help=_("Specify formula tracing."))
    parser.add_option("--formulaParamInputValue", action="store_true", dest="formulaParamInputValue", help=_("Specify formula tracing."))
    parser.add_option("--formulaCallExprSource", action="store_true", dest="formulaCallExprSource", help=_("Specify formula tracing."))
    parser.add_option("--formulaCallExprCode", action="store_true", dest="formulaCallExprCode", help=_("Specify formula tracing."))
    parser.add_option("--formulaCallExprEval", action="store_true", dest="formulaCallExprEval", help=_("Specify formula tracing."))
    parser.add_option("--formulaCallExprResult", action="store_true", dest="formulaCallExprResult", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarSetExprEval", action="store_true", dest="formulaVarSetExprEval", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarSetExprResult", action="store_true", dest="formulaVarSetExprResult", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarSetTiming", action="store_true", dest="timeVariableSetEvaluation", help=_("Specify showing times of variable set evaluation."))
    parser.add_option("--formulaAsserResultCounts", action="store_true", dest="formulaAsserResultCounts", help=_("Specify formula tracing."))
    parser.add_option("--formulaFormulaRules", action="store_true", dest="formulaFormulaRules", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarsOrder", action="store_true", dest="formulaVarsOrder", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarExpressionSource", action="store_true", dest="formulaVarExpressionSource", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarExpressionCode", action="store_true", dest="formulaVarExpressionCode", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarExpressionEvaluation", action="store_true", dest="formulaVarExpressionEvaluation", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarExpressionResult", action="store_true", dest="formulaVarExpressionResult", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarFilterWinnowing", action="store_true", dest="formulaVarFilterWinnowing", help=_("Specify formula tracing."))
    parser.add_option("--formulaVarFiltersResult", action="store_true", dest="formulaVarFiltersResult", help=_("Specify formula tracing."))
    parser.add_option("--uiLang", action="store", dest="uiLang",
                      help=_("Language for user interface (override system settings, such as program messages).  Does not save setting."))
    parser.add_option("--uilang", action="store", dest="uiLang", help=SUPPRESS_HELP)
    parser.add_option("--proxy", action="store", dest="proxy",
                      help=_("Modify and re-save proxy settings configuration.  " 
                             "Enter 'system' to use system proxy setting, 'none' to use no proxy, "
                             "'http://[user[:password]@]host[:port]' "
                             " (e.g., http://192.168.1.253, http://example.com:8080, http://joe:secret@example.com:8080), "
                             " or 'show' to show current setting, ."))
    parser.add_option("--internetConnectivity", choices=("online", "offline"), dest="internetConnectivity",
                      help=_("Specify internet connectivity: online or offline"))
    parser.add_option("--internetconnectivity", action="store", dest="internetConnectivity", help=SUPPRESS_HELP)
    parser.add_option("--internetTimeout", type="int", dest="internetTimeout",
                      help=_("Specify internet connection timeout in seconds (0 means unlimited)."))
    parser.add_option("--internettimeout", type="int", action="store", dest="internetTimeout", help=SUPPRESS_HELP)
    parser.add_option("--plugins", action="store", dest="plugins",
                      help=_("Modify plug-in configuration.  "
                             "Re-save unless 'temp' is in the module list.  " 
                             "Enter 'show' to show current plug-in configuration.  "
                             "Commands show, and module urls are '|' separated: "
                             "+url to add plug-in by its url or filename, ~name to reload a plug-in by its name, -name to remove a plug-in by its name, "
                             "relative URLs are relative to installation plug-in directory, "
                             " (e.g., '+http://arelle.org/files/hello_web.py', '+C:\Program Files\Arelle\examples\plugin\hello_dolly.py' to load, "
                             "or +../examples/plugin/hello_dolly.py for relative use of examples directory, "
                             "~Hello Dolly to reload, -Hello Dolly to remove).  "
                             "If + is omitted from .py file nothing is saved (same as temp).  "
                             "Packaged plug-in urls are their directory's url.  "))
    parser.add_option("--abortOnMajorError", action="store_true", dest="abortOnMajorError", help=_("Abort process on major error, such as when load is unable to find an entry or discovered file."))
    parser.add_option("--collectProfileStats", action="store_true", dest="collectProfileStats", help=_("Collect profile statistics, such as timing of validation activities and formulae."))
    parser.add_option("--debugMode", action="store_true", dest="debugMode", help=_("Let the debugger handle exceptions."))
    # moved here by wch even though 
    try:
        from arelle import webserver 
        hasWebServer = True
        webserver.__name__
    except ImportError:
        hasWebServer = False
    if hasWebServer:
        parser.add_option("--webserver", action="store", dest="webserver",
                          help=_("start web server on host:port[:server] for REST and web access, e.g., --webserver locahost:8080, "
                                 "or specify nondefault a server name, such as cherrypy, --webserver locahost:8080:cherrypy. "
                                 "(It is possible to specify options to be defaults for the web server, such as disclosureSystem and validations, but not including file names.) "))
    
    pluginOptionsIndex = len(parser.option_list)

    # install any dynamic plugins so their command line options can be parsed if present
    for i, arg in enumerate(args):
        if arg.startswith('--plugins'):
            if len(arg) > 9 and arg[9] == '=':
                preloadPlugins = arg[10:]
            elif i < len(args) - 1:
                preloadPlugins = args[i + 1]
            else:
                preloadPlugins = ""
            for pluginCmd in preloadPlugins.split('|'):
                cmd = pluginCmd.strip()
                if cmd not in ("show", "temp") and len(cmd) > 0 and cmd[0] not in ('-', '~', '+'):
                    moduleInfo = PluginManager.addPluginModule(cmd)
                    if moduleInfo:
                        controller.preloadedPlugins[cmd] = moduleInfo
                        PluginManager.reset()
            break
    # add plug-in options
    for optionsExtender in PluginManager.pluginClassMethods("CntlrCmdLine.Options"):
        optionsExtender(parser)
    # pluginLastOptionIndex = len(parser.option_list)
    parser.add_option("-a", "--about",
                      action="store_true", dest="about",
                      help=_("Show product version, copyright, and license."))
    if args is None and controller.isGAE:
        args = ["--webserver=::gae"]
    elif controller.isMSW:
        # if called from java on Windows any empty-string arguments are lost, see:
        # http://bugs.sun.com/view_bug.do?bug_id=6518827
        # insert needed arguments
        args = []
        namedOptions = set()
        optionsWithArg = set()
        for option in parser.option_list:
            names = str(option).split('/')
            namedOptions.update(names)
            # print(names,"  ",namedOptions.update(names))
            if option.action == "store":
                optionsWithArg.update(names)
        priorArg = None
        helpmode = 0
        for arg in sys.argv[1:]:
            if priorArg in optionsWithArg and arg in namedOptions:
                # probable java/MSFT interface bug 6518827
                args.append('')  # add empty string argument
            if arg == "--help":
                helpmode = 1
            args.append(arg)
            priorArg = arg
        if helpmode == 1:
            print(_("Arelle(r) {0}").format(Version.version))
        
    (options, leftoverArgs) = parser.parse_args(args)
    isOk = handleLeftoverOptions(controller, parser, options, leftoverArgs, pluginOptionsIndex)
    return options, isOk

def handleLeftoverOptions(controller, parser, options, leftoverArgs, pluginOptionsIndex):    
    # Handle options that perform a simple one-time function then exit
    # Also, startwebserver if that option is available or initialize logging for renderer daemon or single instance    
    pluginLastOptionIndex = len(parser.option_list)
    if options.about:
        print(_("\narelle(r) {0}bit {1}\n\n"
                "An open source XBRL platform\n"
                "(c) 2010-2014 Mark V Systems Limited\n"
                "All rights reserved\nhttp://www.arelle.org\nsupport@arelle.org\n\n"
                "Licensed under the Apache License, Version 2.0 (the \"License\"); "
                "you may not \nuse this file except in compliance with the License.  "
                "You may obtain a copy \nof the License at "
                "'http://www.apache.org/licenses/LICENSE-2.0'\n\n"
                "Unless required by applicable law or agreed to in writing, software \n"
                "distributed under the License is distributed on an \"AS IS\" BASIS, \n"
                "WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  \n"
                "See the License for the specific language governing permissions and \n"
                "limitations under the License."
                "\n\nIncludes:"
                "\n   Python(r) {3[0]}.{3[1]}.{3[2]} (c) 2001-2013 Python Software Foundation"
                "\n   PyParsing (c) 2003-2013 Paul T. McGuire"
                "\n   lxml {4[0]}.{4[1]}.{4[2]} (c) 2004 Infrae, ElementTree (c) 1999-2004 by Fredrik Lundh"
                "\n   xlrd (c) 2005-2013 Stephen J. Machin, Lingfo Pty Ltd, (c) 2001 D. Giffin, (c) 2000 A. Khan"
                "\n   xlwt (c) 2007 Stephen J. Machin, Lingfo Pty Ltd, (c) 2005 R. V. Kiseliov"
                "{2}"
                ).format(controller.systemWordSize, Version.version,
                         "",
                         sys.version_info, etree.LXML_VERSION))
    elif options.disclosureSystemName in ("help", "help-verbose"):
        text = _("Disclosure system choices: \n{0}").format(' \n'.join(controller.modelManager.disclosureSystem.dirlist(options.disclosureSystemName)))
        try:
            print(text)
        except UnicodeEncodeError:
            print(text.encode("ascii", "replace").decode("ascii"))
            
    elif options.showOptions:
        controller.startLogging(logFileName=(options.logFile or "logToPrint"),
                           logFormat=(options.logFormat or "%(asctime)s [%(messageCode)s] %(message)s - %(file)s"),
                           logLevel=(options.logLevel or "DEBUG"))        
        for optName, optValue in sorted(options.__dict__.items(), key=lambda optItem: optItem[0]):
            controller.logInfo("Option {}={}".format(optName, optValue))

        configFileTemp = IoManager.absPathOnPythonPath(controller,options.configFile)
        controller.logDebug("Adding contents of the Edgar renderer configuration file, '{}', to log file.".format(configFileTemp))
        with open(configFileTemp, "r") as ins:
            for line in ins:
                controller.logInfo(line.strip())        
        controller.logInfo("sys.argv {0}".format(sys.argv))
    
    elif options.showErrors:
        controller.startLogging(logFileName=(options.logFile or "logToPrint"),
                           logFormat=(options.logFormat or "%(asctime)s [%(messageCode)s] %(message)s - %(file)s"),
                           logLevel=(options.logLevel or "DEBUG"))        
        print("Adding list of Edgar renderer error and warning messages to log file.")
              
    elif len(leftoverArgs) != 0 or (options.entrypoint is None and options.configFile is None and 
                                    ((not options.proxy) and (not options.plugins) and
                                     (not any(pluginOption for pluginOption in parser.option_list[pluginOptionsIndex:pluginLastOptionIndex])) and
                                     (options.webserver is None))):
        #message = ErrorMgr.getError('INCORRECT_ARGUMENTS')
        parser.error("Incorrect arguments, please try python EdgarRenderer.py --help")
    
    elif options.webserver:
        # webserver incompatible with file operations
        if any((options.entrypoint, options.importFiles, options.diffFile, options.versReportFile,
                options.factsFile, options.factListCols, options.factTableFile,
                options.conceptsFile, options.preFile, options.calFile, options.dimFile, options.formulaeFile, options.viewArcrole, options.viewFile,
                options.roleTypesFile, options.arcroleTypesFile
                )):
            #message = ErrorMgr.getError('INCORRECT_WEBSRV_ARGUMENTS')
            parser.error("Incorrect arguments with --webserver, please try python EdgarRenderer.py --help")
        else:
            controller.startLogging(logFileName='logToBuffer')
            from arelle import CntlrWebMain
            CntlrWebMain.startWebserver(controller, options)
    
    else:  # The arguments are fine, we can begin logging then parse and run the FILENAME.        
        controller.startLogging(logFileName=(options.logFile or "logToPrint"),
                           logFormat=(options.logFormat or "%(asctime)s [%(messageCode)s] %(message)s - %(file)s"),
                           logLevel=(options.logLevel or "DEBUG"))
   
        return True
    
    return False



class EdgarRenderer(Cntlr.Cntlr):
    """
    .. class:: EdgarRenderer()
    
    Initialization sets up for platform via Cntlr.Cntlr.
    """
    
    def __init__(self, logFileName=None):
        super(EdgarRenderer, self).__init__(hasGui=False)
        self.preloadedPlugins = {}
        self.ErrorMsgs = []
        self.entrypoint = None  # Contains the absolute path of the instance, inline, zip, or folder.
        self.entrypointFolder = None  # Contains absolute folder of instance, inline, or zip; equal to entrypoint if a folder.
        self.nextFileNum = 1 # important for naming file numbers for multi-instance filings
        self.nextUncategorizedFileNum = 9999
        self.nextBarChartFileNum = 0
        self.xlWriter = None
        self.excelXslt = None
        self.createdFolders = []

    def processShowOptions(self, options):
        if options.showOptions:  # debug options
            for optName, optValue in sorted(options.__dict__.items(), key=lambda optItem: optItem[0]):
                self.logInfo(_("Option {0}={1}").format(optName, optValue))
            IoManager.logConfigFile(self, options, messageCode='info')
    
    
    def processMiscOptions(self, options):
        if options.uiLang:  # set current UI Lang (but not config setting)
            self.setUiLanguage(options.uiLang)
        if options.proxy:
            if options.proxy != "show":
                proxySettings = WebCache.proxyTuple(options.proxy)
                self.webCache.resetProxies(proxySettings)
                self.config["proxySettings"] = proxySettings
                self.saveConfig()
                self.logInfo(_("Proxy configuration has been set."))
            useOsProxy, urlAddr, urlPort, user, password = self.config.get("proxySettings", WebCache.proxyTuple("none"))
            if useOsProxy:
                self.logDebug(_("Proxy configured to use {0}.").format(
                    _('Microsoft Windows Internet Settings') if sys.platform.startswith("win")
                    else (_('Mac OS X System Configuration') if sys.platform in ("darwin", "macos")
                    else _('environment variables'))))
            elif urlAddr:
                self.logDebug(_("Proxy setting: http://{0}{1}{2}{3}{4}").format(
                    user if user else "",
                    ":****" if password else "",
                    "@" if (user or password) else "",
                    urlAddr,
                    ":{0}".format(urlPort) if urlPort else ""))
            else:
                self.logDebug(_("Proxy is disabled."))
                    
        if options.plugins:
            resetPlugins = False
            savePluginChanges = True
            showPluginModules = False
            for pluginCmd in options.plugins.split('|'):
                cmd = pluginCmd.strip()
                if cmd == "show":
                    showPluginModules = True
                elif cmd == "temp":
                    savePluginChanges = False
                elif cmd.startswith("+"):
                    moduleInfo = PluginManager.addPluginModule(cmd[1:])
                    if moduleInfo:
                        self.logDebug(_("Addition of plug-in {0} successful.").format(moduleInfo.get("name")),
                                        file=moduleInfo.get("moduleURL"))
                        resetPlugins = True
                        if "CntlrCmdLine.Options" in moduleInfo["classMethods"]:
                            addedPluginWithCntlrCmdLineOptions = True
                            ignore = addedPluginWithCntlrCmdLineOptions                 
                    else:
                        self.addToLog(_("Unable to load plug-in."), messageCode="info", file=cmd[1:])
                elif cmd.startswith("~"):
                    if PluginManager.reloadPluginModule(cmd[1:]):
                        self.addToLog(_("Reload of plug-in successful."), messageCode="info", file=cmd[1:])
                        resetPlugins = True
                    else:
                        self.addToLog(_("Unable to reload plug-in."), messageCode="info", file=cmd[1:])
                elif cmd.startswith("-"):
                    if PluginManager.removePluginModule(cmd[1:]):
                        self.addToLog(_("Deletion of plug-in successful."), messageCode="info", file=cmd[1:])
                        resetPlugins = True
                    else:
                        self.addToLog(_("Unable to delete plug-in."), messageCode="info", file=cmd[1:])
                else:  # assume it is a module or package (may also have been loaded before for option parsing)
                    savePluginChanges = False
                    if cmd in self.preloadedPlugins:
                        moduleInfo = self.preloadedPlugins[cmd]  # already loaded, add activation message to log below
                    else:
                        moduleInfo = PluginManager.addPluginModule(cmd)
                        if moduleInfo:
                            resetPlugins = True
                    if moduleInfo: 
                        self.addToLog(_("Activation of plug-in {0} successful.").format(moduleInfo.get("name")),
                                        messageCode="info", file=moduleInfo.get("moduleURL"))
                        resetPlugins = True
                    else:
                        self.addToLog(_("Unable to load {0} as a plug-in or {0} is not recognized as a command. ").format(cmd), messageCode="info", file=cmd)
                if resetPlugins:
                    PluginManager.reset()
                    if savePluginChanges:
                        PluginManager.save(self)
            if showPluginModules:
                self.addToLog(_("Plug-in modules:"), messageCode="info")
                for i, moduleItem in enumerate(sorted(PluginManager.pluginConfig.get("modules", {}).items())):
                    i  # get rid of warning on unused i
                    moduleInfo = moduleItem[1]
                    self.addToLog(_("Plug-in: {0}; author: {1}; version: {2}; status: {3}; date: {4}; description: {5}; license {6}.").format(
                                    moduleItem[0], moduleInfo.get("author"), moduleInfo.get("version"), moduleInfo.get("status"),
                                    moduleInfo.get("fileDate"), moduleInfo.get("description"), moduleInfo.get("license")),
                                    messageCode="info", file=moduleInfo.get("moduleURL"))
    
    
    def retrieveDefaultREConfigParams(self, options):        
        # defaultValueDict contains the default strings for
        # when there are no config file and no command line arguments.
        self.defaultValueDict = defaultdict(lambda:None)
        self.defaultValueDict['abortOnMajorError'] = str(True)
        self.defaultValueDict['archiveFolder'] = 'Archive'
        self.defaultValueDict['auxMetadata'] = str(False)
        self.defaultValueDict['copyInlineFilesToOutput'] = str(False)
        self.defaultValueDict['deleteProcessedFilings'] = str(True)
        self.defaultValueDict['deliveryFolder'] = 'Delivery'
        self.defaultValueDict['debugMode'] = str(False)
        self.defaultValueDict['errorsFolder'] = 'Errors'
        self.defaultValueDict['excelXslt'] = 'InstanceReport_XmlWorkbook.xslt'
        self.defaultValueDict['failFile'] = 'errorLog.log'
        self.defaultValueDict['filingsFolder'] = 'Filings'
        self.defaultValueDict['htmlReportFormat'] = 'Complete'
        self.defaultValueDict['internetConnectivity'] = 'offline' 
        self.defaultValueDict['noEquity'] = str(False)
        self.defaultValueDict['processingFolder'] = 'Processing'
        self.defaultValueDict['processingFrequency'] = '10'
        self.defaultValueDict['renderingService'] = 'Instance'
        self.defaultValueDict['reportFormat'] = 'Html'
        self.defaultValueDict['reportsFolder'] = 'Reports'
        self.defaultValueDict['reportXslt'] = 'InstanceReport.xslt'
        self.defaultValueDict['resourcesFolder'] = '..\\resources'
        self.defaultValueDict['saveTargetInstance'] = str(False)
        self.defaultValueDict['saveTargetFiling'] = str(False)
        self.defaultValueDict['sourceList'] = ''
        self.defaultValueDict['summaryXslt'] = None
        self.defaultValueDict['totalClean'] = str(False)
        self.defaultValueDict['utrValidate'] = str(False)
        self.defaultValueDict['validate'] = str(False)
        self.defaultValueDict['validateEFM'] = str(False)
        self.defaultValueDict['zipOutputFile'] = None
        
        # The configDict holds the values as they were read from the config file.
        # Only options that appear with a value that is not None in defaultValueDict are recognized.
        self.configDict = defaultdict(lambda:None)
        configLocation = IoManager.getConfigFile(self,options)
        if configLocation is None: # Although it is odd not to have a config file, it is not an error.
            self.logDebug(_("No config file"))
        else:
            self.logDebug(_("Extracting info from config file {}".format(configLocation)))
            tree = etree.parse(configLocation)
            for child in tree.iter():
                if child.tag is not etree.Comment and child.text is not None:
                    if child.tag in self.defaultValueDict:
                        value = child.text.strip()
                        if value == '': value = None
                        self.configDict[child.tag] = value
                    elif len(child.text.strip()) > 0: # Extra tags with no content are ignored
                        #message = ErrorMgr.getError('UNSUPPORTED_CONFIG_TAG').format(child.text, child.tag)
                        self.logWarn("Found value {} for unsupported configuration tag {}".format(child.text, child.tag)) 

    def initializeReOptions(self, options):
        self.logDebug("General options:")
        
        def setProp(prop, init, rangeList=None, cs=False):
            value = next((x
                         for x in [init, self.configDict[prop], self.defaultValueDict[prop]]
                         if x is not None), None)
            if type(value)==str and not cs: value = value.casefold()
            setattr(self, prop, value)
            if rangeList is not None and value not in [x.casefold() for x in rangeList]:
                raise Exception("Unknown {} '{}' not in {} (case insensitive) on command line or config file.".format(prop,value,rangeList))
            self.logDebug("{}=\t{}".format(prop, value))            
            return value
        
        # options applicable to rendering in either mode: 
        options.renderingService = setProp('renderingService', options.renderingService, rangeList=['Instance','Daemon'])        
        options.reportFormat = setProp('reportFormat', options.reportFormat, rangeList=['Html', 'Xml', 'HtmlAndXml'])               
        options.htmlReportFormat = setProp('htmlReportFormat', options.htmlReportFormat, rangeList=['Complete','Fragment'])
        options.zipOutputFile = setProp('zipOutputFile', options.zipOutputFile,cs=True)    
        options.sourceList = " ".join(setProp('sourceList', options.sourceList,cs=True).split()).split(',')
        self.sourceDict={}
        # Parse comma and colon separated list a:b b:c, d:e:f into a dictionary {'a': ('b b','c'), 'd': ('e','f') }:
        for source in options.sourceList:
            if (len(source) > 0):
                s = source.split(':') # we must accomodate spaces in tokens separated by colons
                if (len(s) != 3):
                    self.logWarn("Ignoring bad token {} in {}".format(s,options.sourceList))
                else: # we do not accomodate general URL's in the 'original' field.
                    instance = " ".join(s[0].split())
                    doctype = " ".join(s[1].split())
                    original = " ".join(s[2].split())
                    self.sourceDict[instance]=(doctype,original)
                
        # These options have to be passed back to arelle via the options object
        options.internetConnectivity = setProp('internetConnectivity',options.internetConnectivity, rangeList=['online','offline'])
        
        def setFlag(flag, init):
            setattr(self, flag, next((Utils.booleanFromString(x) 
                             for x in [init
                                       , self.configDict[flag], self.defaultValueDict[flag]]
                             if x is not None), None))
            self.logDebug("{}=\t{}".format(flag, getattr(self, flag)))
            return getattr(self, flag)
            
        options.abortOnMajorError = setFlag('abortOnMajorError', options.abortOnMajorError)
        options.totalClean = setFlag('totalClean', options.totalClean)
        options.noEquity = setFlag('noEquity', options.noEquity)
        options.auxMetadata = setFlag('auxMetadata', options.auxMetadata)
        options.copyInlineFilesToOutput = setFlag('copyInlineFilesToOutput', options.copyInlineFilesToOutput)
        options.saveTargetInstance = setFlag('saveTargetInstance',options.saveTargetInstance)
        options.saveTargetFiling = setFlag('saveTargetFiling',options.saveTargetFiling)      
        # note that delete processed filings is only relevant when the input had to be unzipped.
        options.deleteProcessedFilings = setFlag('deleteProcessedFilings', options.deleteProcessedFilings)
        options.debugMode = setFlag('debugMode', options.debugMode)        
        # These flags have to be passed back to arelle via the options object.
        options.validate = setFlag('validate', options.validate)
        options.utrValidate = setFlag('utrValidate', options.utrValidate)
        options.validateEFM = setFlag('validateEFM', options.validateEFM)
    
        
        def setFolder(folder, init, searchPythonPath=False):
            if searchPythonPath: # if the folder is not an absolute path, we want to look for it relative to the python path.
                value= next((IoManager.absPathOnPythonPath(self,x)
                            for x in [init, self.configDict[folder], self.defaultValueDict[folder]]
                                 if x is not None), None)
            else: # otherwise interpret the folder as being relative to some subsequent processing location.
                value = next((x
                                 for x in [init, self.configDict[folder], self.defaultValueDict[folder]]
                                 if x is not None), None)
            setattr(self, folder, value)
            self.logDebug("{}=\t{}".format(folder, getattr(self, folder)))
            return getattr(self, folder)
        
        options.processingFolder = setFolder('processingFolder', options.processingFolder)
        options.reportsFolder = setFolder('reportsFolder', options.reportsFolder)
        options.resourcesFolder = setFolder('resourcesFolder', options.resourcesFolder,searchPythonPath=True)


        def setResourceFile(file, init, errstr):
            setattr(self, file, None)
            value = next((x
                         for x in [init, self.configDict[file], self.defaultValueDict[file]]
                         if x is not None), None)
            if value is not None and type(value)==str and len(value)>0:                
                if not isdir(self.resourcesFolder):
                    # It is possible to specify a bad resources folder, but then not actually need it;
                    # that is why the test for its presence is here and not earlier.
                    raise Exception("Cannot locate resources folder '{}'".format(self.resourcesFolder))
                value = os.path.join(self.resourcesFolder,value)
                setattr(self, file, value)
                if getattr(self, file) is not None and not isfile(getattr(self, file)):
                    raise Exception(_(errstr).format(self.reportXslt))
            self.logDebug("{}=\t{}".format(file, value))            
            return value
       
        #setResourceFile('reportXslt', options.reportXslt, 'INVALID_CONFIG_REPORTXSLT')
        setResourceFile('reportXslt', options.reportXslt, "Cannot find report xslt {}")
        # Report XSLT is required when reportFormat contains 'Html'.     
        if self.reportXslt is None and 'html' in self.reportFormat.casefold():
            raise Exception('No {} specified when {}={} requires it.'.format('reportXslt', 'reportFormat', self.reportFormat))

        # Summary XSLT is optional, but do report if you can't find it.
        #setResourceFile('summaryXslt', options.summaryXslt, 'INVALID_CONFIG_SUMMARYXSLT')
        setResourceFile('summaryXslt', options.summaryXslt, "Cannot find summary xslt {}")

        # Excel XSLT is optional, but do report if you can't find it.
        #setResourceFile('excelXslt', options.excelXslt, 'INVALID_CONFIG_EXCELXSLT')  
        setResourceFile('excelXslt', options.excelXslt, "Cannot find excel xslt {}")

        
    def initializeReSinglesOptions(self, options):
        # At the moment there are not any options relevant only for single-instance mode.
        return
    
    
    def initializeReDaemonOptions(self, options):
        self.logDebug("Daemon Options:")
        
        def setLocation(location, init, mustExist=False, isfile=False):
            value = next((os.path.join(os.getcwd(), x)
                             for x in [init
                                       , self.configDict[location], self.defaultValueDict[location]]
                             if x is not None), None)
            setattr(self, location, value)
            self.logDebug("{}=\t{}".format(location, value))
            if mustExist and value is None:
                raise Exception('No {} specified for Daemon.'.format(location))
            if mustExist and not isfile and not isdir(value):
                raise Exception('No such {} as {}.'.format(location, value))
            return value
        
        setLocation('filingsFolder', options.filingsFolder)
        setLocation('deliveryFolder', options.deliveryFolder)
        setLocation('errorsFolder', options.errorsFolder)
        setLocation('archiveFolder', options.archiveFolder)
        setLocation('failFile', options.failFile)
       
        def setProp(prop, init, required=False):
            value = next((x
                         for x in [init, self.configDict[prop], self.defaultValueDict[prop]]
                         if x is not None), None)
            setattr(self, prop, value)
            if required and value is None:
                raise Exception('{} not set for daemon'.format(prop))        
            self.logDebug("{}=\t{}".format(prop, getattr(self, prop)))
            return value
        
        setProp('processingFrequency', options.processingFrequency, required=True)
        
       

    @property
    def renderMode(self):
        return self.renderingService.casefold()
    @property
    def isSingles(self):
        return self.renderingService.casefold() == 'instance'
    @property
    def isDaemon(self):
        return self.renderingService.casefold() == 'daemon'
    
    def initializeModelManager(self, options):
        if options.validateEFM:
            if options.disclosureSystemName:
                self.logDebug(_("both --efm and --disclosureSystem validation are requested, proceeding with --efm only"))
            self.modelManager.validateDisclosureSystem = True
            self.modelManager.disclosureSystem.select("efm")
        elif options.disclosureSystemName:
            self.modelManager.validateDisclosureSystem = True
            self.modelManager.disclosureSystem.select(options.disclosureSystemName)
        else:
            self.modelManager.disclosureSystem.select(None)  # just load ordinary mappings
            self.modelManager.validateDisclosureSystem = False
        if options.utrUrl:  # override disclosureSystem utrUrl
            self.modelManager.disclosureSystem.utrUrl = options.utrUrl
        # can be set now because the utr is first loaded at validation time 
    
        # disclosure system sets logging filters, override disclosure filters, if specified by command line
        if options.logLevelFilter:
            self.setLogLevelFilter(options.logLevelFilter)
        if options.logCodeFilter:
            self.setLogCodeFilter(options.logCodeFilter)
        if options.calcDecimals:
            if options.calcPrecision:
                self.logDebug(_("both --calcDecimals and --calcPrecision validation are requested, proceeding with --calcDecimals only"))
            self.modelManager.validateInferDecimals = True
            self.modelManager.validateCalcLB = True
        elif options.calcPrecision:
            self.modelManager.validateInferDecimals = False
            self.modelManager.validateCalcLB = True
        if options.utrValidate:
            self.modelManager.validateUtr = True
        if options.infosetValidate:
            self.modelManager.validateInfoset = True
        if  options.abortOnMajorError:
            self.modelManager.abortOnMajorError = True
        if options.collectProfileStats:
            self.modelManager.collectProfileStats = True
        if (options.internetConnectivity or "").casefold() == "offline":
            self.webCache.workOffline = True
        elif (options.internetConnectivity or "").casefold() == "online":
            self.webCache.workOffline = False
        if options.internetTimeout is not None:
            self.webCache.timeout = (options.internetTimeout or None)  # use None if zero specified to disable timeout
        fo = ModelFormulaObject.FormulaOptions()
        if options.parameters:
            parameterSeparator = (options.parameterSeparator or ',')
            fo.parameterValues = dict(((ModelValue.qname(key, noPrefixIsNoNamespace=True), (None, value)) 
                                   for param in options.parameters.split(parameterSeparator) 
                                   for key, sep, value in (param.partition('='),)))   
        if options.formulaParamExprResult:
            fo.traceParameterExpressionResult = True
        if options.formulaParamInputValue:
            fo.traceParameterInputValue = True
        if options.formulaCallExprSource:
            fo.traceCallExpressionSource = True
        if options.formulaCallExprCode:
            fo.traceCallExpressionCode = True
        if options.formulaCallExprEval:
            fo.traceCallExpressionEvaluation = True
        if options.formulaCallExprResult:
            fo.traceCallExpressionResult = True
        if options.formulaVarSetExprEval:
            fo.traceVariableSetExpressionEvaluation = True
        if options.formulaVarSetExprResult:
            fo.traceVariableSetExpressionResult = True
        if options.formulaAsserResultCounts:
            fo.traceAssertionResultCounts = True
        if options.formulaFormulaRules:
            fo.traceFormulaRules = True
        if options.formulaVarsOrder:
            fo.traceVariablesOrder = True
        if options.formulaVarExpressionSource:
            fo.traceVariableExpressionSource = True
        if options.formulaVarExpressionCode:
            fo.traceVariableExpressionCode = True
        if options.formulaVarExpressionEvaluation:
            fo.traceVariableExpressionEvaluation = True
        if options.formulaVarExpressionResult:
            fo.traceVariableExpressionResult = True
        if options.timeVariableSetEvaluation:
            fo.timeVariableSetEvaluation = True
        if options.formulaVarFilterWinnowing:
            fo.traceVariableFilterWinnowing = True
        if options.formulaVarFiltersResult:
            fo.traceVariableFiltersResult = True
        if options.formulaVarFiltersResult:
            fo.traceVariableFiltersResult = True
        return fo
    
    
    
    def dequeueInputZip(self, options):  # returns the location of zip ready to unpack
        # upon exit, options.entrypoint is set to absolute location of the zip file after the move,
        # and self.processingFolder is set to absolute location of where it should be processed.
        inputFileSource = None
        zipfound = False  
        # check for next filing in input folder
        # as usual, self.{some}Folder is absolute, while options.{some}Folder is what was specified in the input.
        self.originalProcessingFolder = os.path.join(getcwd(), self.processingFolder)    
        self.logDebug(_("Checking for the oldest zip file in {}").format(self.filingsFolder))            
        while not zipfound:
            for file in sorted(os.listdir(self.filingsFolder), key=lambda file: os.stat(join(self.filingsFolder, file)).st_mtime):
                if not zipfound and Utils.isZipFilename(file):
                    inputFileSource = join(self.filingsFolder, file)      
                    self.processingFolder = IoManager.createNewFolder(self,self.originalProcessingFolder,file)
                    # reportsFolder = normpath(join(self.processingFolder,options.reportsFolder)) 
                    processingFileSource = join(self.processingFolder, file)
                    if not exists(inputFileSource): continue  # it got deleted before we could process it.
                    self.logDebug(_("Found a new zip file to process; moving {} to Processing folder ").format(inputFileSource))                   
                    try:
                        IoManager.move_clobbering_file(inputFileSource, processingFileSource)
                        options.entrypoint = processingFileSource
                        zipfound = True              
                    except IOError as err: 
                        self.logError(str(err))
                        #self.logError(_(ErrorMgr.getError('FILING_MOVE_ERROR').format(self.processingFolder)))              
                        self.logError(_("Could not remove {}").format(self.processingFolder))              
                        try: removedirs(self.processingFolder)
                        except IOError: continue
                    self.zipOutputFile = file
                    self.doneFile = join(self.archiveFolder, file)
                    # self.failFile = join(self.errorsFolder,file)
                    break
            # no more files.
            if not zipfound:
                sleep = self.processingFrequency
                self.logDebug(_("Sleeping for " + sleep + " seconds. "))
                time.sleep(float(sleep))    
        return
    
   
    def loadModel(self, options, inputFileSource):  # success, modelXbrl, firstStartedAt, modelDiffReport, formulaOptions
                
        formulaOptions = self.initializeModelManager(options)
        self.modelManager.formulaOptions = formulaOptions
        timeNow = XmlUtil.dateunionValue(datetime.datetime.now())
        firstStartedAt = startedAt = time.time()
        modelDiffReport = None
        modelXbrl = None
        try:
            if inputFileSource:
                self.logDebug(_("Initializing modelXbrl: "))
                modelXbrl = self.modelManager.load(inputFileSource, _("views loading"))
                success = True
                
        except ModelDocument.LoadingException as err:
            self.logInfo(_("Loading exception: {!s}".format(err)))
            success = not self.modelManager.abortOnMajorError
    
        except Exception as err:
            self.logError(_('[Exception] Failed to complete request: {} {}'.format(err, traceback.format_tb(sys.exc_info()[2]))))
            success = False  # loading errors, don't attempt to utilize loaded DTS
    
        if success and modelXbrl and modelXbrl.modelDocument:
            self.logDebug("Loading modelXBRL profileStat: ")
            loadTime = time.time() - startedAt
            modelXbrl.profileStat(_("load"), loadTime)
            self.logDebug(Locale.format_string(self.modelManager.locale,
                                        _("DTS loaded in %.2f secs at %s"),
                                        (loadTime, timeNow)))
            if options.importFiles:
                self.logDebug(_("Handling import files "))
                for importFile in options.importFiles.split("|"):
                    fileName = importFile.strip()
                    if not (Utils.isHttpFilename(fileName) or os.path.isabs(fileName)): 
                        fileName = dirname(modelXbrl.uri) + os.sep + fileName 
                    ModelDocument.load(modelXbrl, fileName)
                    loadTime = time.time() - startedAt
                    self.logDebug(Locale.format_string(self.modelManager.locale,
                                            _("Additional DTS imported in %.2f secs at %s"),
                                            (loadTime, timeNow)))
                    modelXbrl.profileStat(_("import"), loadTime)
                if modelXbrl.errors:
                    #message = ErrorMgr.getError('DTS_LOADING_IMPORT_ERROR').format(err, traceback.format_tb(sys.exc_info()[2]))
                    self.logDebug(_("Model import errors found: {} {}".format(err, traceback.format_tb(sys.exc_info()[2]))))
                    success = False  # loading errors, don't attempt to utilize loaded DTS
            if modelXbrl.modelDocument.type in ModelDocument.Type.TESTCASETYPES:
                self.logDebug("Loading modelXbrl TestCaseTypes")
                for pluginXbrlMethod in PluginManager.pluginClassMethods("Testcases.Start"):
                    pluginXbrlMethod(self, options, modelXbrl)
            else:  # not a test case, probably instance or DTS
                self.logDebug("Loading {}".format(inputFileSource))
                for pluginXbrlMethod in PluginManager.pluginClassMethods("CntlrCmdLine.Xbrl.Loaded"):
                    pluginXbrlMethod(self, options, modelXbrl)
        else:
            self.logDebug("No modelXBRL profileStat ", file=inputFileSource)
            success = False
        if success and options.diffFile and options.versReportFile:
            try:
                self.logDebug("Loading diff FileSource ", file=inputFileSource)
                diffFilesource = FileSource.FileSource(options.diffFile, self)
                startedAt = time.time()
                modelXbrl2 = self.modelManager.load(diffFilesource, _("views loading"))
                if modelXbrl2.errors:
                    if not options.keepOpen:
                        modelXbrl2.close()
                    #message = ErrorMgr.getError('DTS_DIFF_FILE_LOADING_ERROR').format(err, traceback.format_tb(sys.exc_info()[2]))
                    self.logDebug("Model Diff-file loading errors found: {} {}".format(err, traceback.format_tb(sys.exc_info()[2])))
                    success = False
                else:
                    loadTime = time.time() - startedAt
                    modelXbrl.profileStat(_("load"), loadTime)
                    self.logDebug(Locale.format_string(self.modelManager.locale,
                                            _("diff comparison DTS loaded in %.2f secs"),
                                            loadTime))
                    startedAt = time.time()
                    modelDiffReport = self.modelManager.compareDTSes(options.versReportFile)
                    diffTime = time.time() - startedAt
                    modelXbrl.profileStat(_("diff"), diffTime)
                    self.logDebug(Locale.format_string(self.modelManager.locale, _("compared in %.2f secs"), diffTime))
            except ModelDocument.LoadingException as err:
                self.logInfo("Loading exception: {}".format(err))
                success = False
            except Exception as err:
                success = False
                #message = ErrorMgr.getError('DTS_DIFF_FILE_LOADING_ERROR').format(err, traceback.format_tb(sys.exc_info()[2]))
                self.logDebug("Model Diff-file loading errors found: {} {}".format(err, traceback.format_tb(sys.exc_info()[2])))
    
        return success, modelXbrl, firstStartedAt, modelDiffReport, formulaOptions
    
    
    def validateInstance(self, options, modelXbrl, formulaOptions):
        success = True
        try:          
            self.logDebug("Loading modelXbrl Formulae (if any) and performing validation")
            modelXbrl = self.modelManager.modelXbrl
            hasFormulae = modelXbrl.hasFormulae
            if self.validate:
                from arelle.Cntlr import LogToBufferHandler
                xbrlErrorLogger = LogToBufferHandler()
                modelXbrl.logger.addHandler(xbrlErrorLogger)
                startedAt = time.time()
                if options.formulaAction:  # don't automatically run formulas
                    modelXbrl.hasFormulae = False
                errorCountBeforeValidation = len(Utils.xbrlErrors(modelXbrl))
                self.modelManager.validate() 
                errorCountAfterValidation = len(Utils.xbrlErrors(modelXbrl))
                errorCountDuringValidation = errorCountAfterValidation - errorCountBeforeValidation
                if errorCountDuringValidation > 0:
                    for record in xbrlErrorLogger.logRecordBuffer.copy(): # copy the list so that errors here do not append to it.
                        try: href = record.refs[0]['href']
                        except: href = ''
                        try: sourceLine = record.refs[0]['sourceLine']
                        except: sourceLine = ''
                        if record.levelno >= logging.ERROR:
                            self.logError(_("[{}] {} - {} {}".format(record.messageCode
                                                                         ,record.getMessage()
                                                                         ,href.split('#')[0]
                                                                         ,sourceLine)))
                    if self.modelManager.abortOnMajorError:
                        success = False
                        self.logFatal(_("Not attempting to render after {} validation errors").format(
                                        errorCountDuringValidation))
                    else:
                        self.logInfo(_("Ignoring {} Validation errors because abortOnMajorError is not set.").format(errorCountDuringValidation))
    
                if options.formulaAction:  # restore setting
                    modelXbrl.hasFormulae = hasFormulae
                self.logDebug(Locale.format_string(self.modelManager.locale,
                                        _("validated in %.2f secs"),
                                        time.time() - startedAt))
                
            if success:
                if options.formulaAction in ("validate", "run"):  # do nothing here if "none"
                    from arelle import ValidateXbrlDimensions, ValidateFormula
                    startedAt = time.time()
                    if not self.validate:
                        ValidateXbrlDimensions.loadDimensionDefaults(modelXbrl)
                    # setup fresh parameters from formula optoins
                    modelXbrl.parameters = formulaOptions.typedParameters()
                    ValidateFormula.validate(modelXbrl, compileOnly=(options.formulaAction != "run"))
                    self.logDebug(Locale.format_string(self.modelManager.locale,
                                        _("formula validation and execution in %.2f secs")
                                        if options.formulaAction == "run"
                                        else _("formula validation only in %.2f secs"),
                                        time.time() - startedAt))
            
                if options.testReport:
                    ViewFileTests.viewTests(modelXbrl, options.testReport, options.testReportCols)
            
                if options.rssReport:
                    ViewFileRssFeed.viewRssFeed(modelXbrl, options.rssReport, options.rssReportCols)
            
                # if options.DTSFile:
                    # ViewFileDTS.viewDTS(modelXbrl, options.DTSFile)
                if options.factsFile:
                    ViewFileFactList.viewFacts(modelXbrl, options.factsFile, labelrole=options.labelRole, lang=options.labelLang, cols=options.factListCols)
                if options.factTableFile:
                    ViewFileFactTable.viewFacts(modelXbrl, options.factTableFile, labelrole=options.labelRole, lang=options.labelLang)
                if options.conceptsFile:
                    ViewFileConcepts.viewConcepts(modelXbrl, options.conceptsFile, labelrole=options.labelRole, lang=options.labelLang)
                if options.preFile:
                    ViewFileRelationshipSet.viewRelationshipSet(modelXbrl, options.preFile, "Presentation Linkbase", "http://www.xbrl.org/2003/arcrole/parent-child", labelrole=options.labelRole, lang=options.labelLang)
                if options.calFile:
                    ViewFileRelationshipSet.viewRelationshipSet(modelXbrl, options.calFile, "Calculation Linkbase", "http://www.xbrl.org/2003/arcrole/summation-item", labelrole=options.labelRole, lang=options.labelLang)
                if options.dimFile:
                    ViewFileRelationshipSet.viewRelationshipSet(modelXbrl, options.dimFile, "Dimensions", "XBRL-dimensions", labelrole=options.labelRole, lang=options.labelLang)
                if options.formulaeFile:
                    ViewFileFormulae.viewFormulae(modelXbrl, options.formulaeFile, "Formulae", lang=options.labelLang)
                if options.viewArcrole and options.viewFile:
                    ViewFileRelationshipSet.viewRelationshipSet(modelXbrl, options.viewFile, basename(options.viewArcrole), options.viewArcrole, labelrole=options.labelRole, lang=options.labelLang)
                if options.roleTypesFile:
                    ViewFileRoleTypes.viewRoleTypes(modelXbrl, options.roleTypesFile, "Role Types", isArcrole=False, lang=options.labelLang)
                if options.arcroleTypesFile:
                    ViewFileRoleTypes.viewRoleTypes(modelXbrl, options.arcroleTypesFile, "Arcrole Types", isArcrole=True, lang=options.labelLang)
                for pluginXbrlMethod in PluginManager.pluginClassMethods("CntlrCmdLine.Xbrl.Run"):
                    pluginXbrlMethod(self, options, modelXbrl)
    
        except (IOError, EnvironmentError) as err:
            #self.logError(ErrorMgr.getError('SAVE_OUTPUT_ERROR').format(err))
            self.logError("Failed to save output: {}".format(err))
            success = False
        except Exception as err:
            success = False
            tb = sys.exc_info()[2]  # t,v,tb tuple
            listOfFrames = []
            n = 0
            while tb is not None: 
                f = tb.tb_frame     
                lineno = tb.tb_lineno     
                co = f.f_code     
                filename = co.co_filename     
                name = co.co_name     
                if filename in linecache.cache:     
                    lines = linecache.cache[filename][2] 
                else:
                    try:
                        lines = linecache.updatecache(filename, f.f_globals)
                    except:
                        lines = []
                if 1 <= lineno <= len(lines):
                    line = lines[lineno - 1]   
                else:
                    line = ''                    
                if line: 
                    line = line.strip()     
                else: 
                    line = "(source not available)"    
                listOfFrames.append((filename, lineno, name, line))     
                tb = tb.tb_next     
                n = n + 1
            text = "Stack trace, most recent frame last:\n"
            for (filename, lineno, name, line) in listOfFrames:
                try:
                    text += "{0}".format(name)
                    text += "\t({0}".format(lineno)
                    text += " in {0})".format(filename)
                    text += "\n\t{0}\n".format(line)
                except:
                    text += "(UNPRINTABLE STACK FRAME)\n"
            #self.logError(_(ErrorMgr.getError('RE3_STACK_FRAME_ERROR')).format(err, text))
            self.logError(_('[Exception] Failed to complete request: {} {}').format(err, text))
        return success, modelXbrl 
    
    
  
    def runRenderer(self, options):
        """Process command line arguments or web service request, such as to load and validate an XBRL document, or start web server.        
        When a web server has been requested, this method may be called multiple times, once for each web service (REST) request that requires processing.
        Otherwise (when called for a command line request) this method is called only once for the command line arguments request.
                   
        :param options: OptionParser options from parse_args of main argv arguments (when called from command line) or corresponding arguments from web service (REST) request.
        :type options: optparse.Values
        """
        self.logDebug("Starting "+VERSION)
        self.logDebug("Command line arguments: {!s}".format(sys.argv))
        # Process command line options
        self.processShowOptions(options)
        self.processMiscOptions(options)                        
        # Run utility command line options that don't depend on entrypoint Files
        hasUtilityPlugin = False
        for pluginXbrlMethod in PluginManager.pluginClassMethods("CntlrCmdLine.Utility.Run"):
            hasUtilityPlugin = True
            try:
                pluginXbrlMethod(self, options, sourceZipStream=None)
            except SystemExit:  # terminate operation, plug in has terminated all processing
                return True  # success
        ignore = hasUtilityPlugin
        
        # Set default config params; overwrite with command line args if necessary
        self.retrieveDefaultREConfigParams(options)
        # Initialize the folders and objects required in both modes.
        self.initializeReOptions(options)
        
        def innerRunRenderer():
            if self.isSingles: # Single mode folders are relative to source file and TEMP if not absolute.
                self.initializeReSinglesOptions(options)
            else:
                # Daemon mode must have at least input, processing, and output folders,
                # but the entrypointFolder and the reportsFolder are created as temporary locations.
                self.initializeReDaemonOptions(options)
                IoManager.handleFolder(self, self.filingsFolder, False, False) 
                IoManager.handleFolder(self, self.deliveryFolder, False, self.totalClean)
                IoManager.handleFolder(self, self.archiveFolder, False, self.totalClean)
                if self.errorsFolder is not None:  # You might not have an errors folder.
                    IoManager.handleFolder(self, self.errorsFolder, False, self.totalClean)             
                self.dequeueInputZip(options)
            if options.entrypoint is None:
                self.logInfo("No filing specified. Exiting renderer.")
                return False
            print(options.entrypoint) # write filename to stdout so user can see what is processed; not needed in the log.
            if not IoManager.unpackInput(self, options): return False
            if zipfile.is_zipfile(options.entrypoint) and self.zipOutputFile is None: 
                # Output zip name same as input by default
                self.zipOutputFile = basename(options.entrypoint)            
            modelXbrl = None        
            self.supplementalFileList = [basename(f) for f in self.supplementList]
            self.instanceSummaryList = []
            self.reportsFolder = join(self.entrypointFolder, self.reportsFolder)
            IoManager.handleFolder(self, self.reportsFolder, True, self.totalClean)
            loopnum = 0
            success = True
            self.logDebug(_("Pre-rendering stats: NumInstance: {!s}; NumInline: {!s}; NumSupplemental: {!s} "
                           ).format(len(self.instanceList), len(self.inlineList), len(self.supplementList)))
            for inputFileSource in sorted(self.instanceList + self.inlineList):
                if success: 
                    loopnum += 1
                    self.entrypoint = inputFileSource
                    (success, modelXbrl, firstStartedAt, modelDiffReport, fo
                     ) = self.loadModel(options, join(self.processingFolder, inputFileSource))
                    self.modelDiffReport = modelDiffReport
                    self.firstStartedAt = firstStartedAt
                    if modelXbrl is not None and self.validate: 
                        (success, modelXbrl) = self.validateInstance(options, modelXbrl, fo)     
                    if success and modelXbrl is not None: 
                        RefManager.RefManager(self.resourcesFolder).loadAddedUrls(modelXbrl, self)  # do this after validation.
                        self.logDebug(_("Start the rendering process on {}, filing loop {!s}.").format(inputFileSource, loopnum))
                        success = Filing.mainFun(self, modelXbrl, self.reportsFolder)
                        self.logDebug(_("End of rendering on {}.").format(inputFileSource))
                        Inline.saveTargetDocumentIfNeeded(self,options,modelXbrl)
            
            if success and modelXbrl:                
                self.postprocessInstance(options, modelXbrl)
                self.logDebug("Post-processing complete")
            return success # from innerRunRenderer
                
        if self.debugMode:
            success = innerRunRenderer()
        else:
            try:
                success = innerRunRenderer()
            except Exception as err:
                self.logError("{} {}".format(err, "")) # traceback.format_tb(sys.exc_info()[2]))) # This takes us into infinite recursion.
                success = False
        if not success:
            self.postprocessFailure(options) 
            self.logDebug("Filing processing complete")
        return success
    
    
    def postprocessInstance(self, options, modelXbrl):
        del modelXbrl.duplicateFactSet
        xlWriter = self.xlWriter
        if xlWriter:
            xlWriter.save()
            xlWriter.close()
            del self.xlWriter 
            self.logDebug("Excel rendering complete")
        modelXbrl.profileStat(_("total"), time.time() - self.firstStartedAt)
        if options.collectProfileStats and modelXbrl:
            modelXbrl.logProfileStats() 
        def copyResourceToReportFolder(filename):
            target = join(self.reportsFolder, filename)
            if not exists(target):
                os.makedirs(self.reportsFolder, exist_ok=True)
                source = join(self.resourcesFolder, filename)
                shutil.copyfile(source, target)                
        if 'html' in (self.reportFormat or "").casefold() or self.summaryXslt is not None:
            copyResourceToReportFolder("Show.js")
            copyResourceToReportFolder("report.css")
        if self.summaryXslt is not None:
            copyResourceToReportFolder("RenderingLogs.xslt")
        # TODO: At this point would be nice to call out any files not loaded in any instance DTS
        inputsToCopyToOutputList = self.supplementList
        if options.copyInlineFilesToOutput: inputsToCopyToOutputList += self.inlineList
        for filename in inputsToCopyToOutputList:
            target = join(self.reportsFolder, filename)
            if exists(target): remove(target)
            source = join(self.processingFolder, filename)
            shutil.copyfile(source, target)                
        self.modelManager.close(modelXbrl)
        self.modelManager.close(self.modelDiffReport)
        self.logDebug("Instance post-processing complete")
        
        summary = Summary.Summary(self)    
        rootETree = summary.buildSummaryETree()
        IoManager.writeXmlDoc(rootETree, os.path.join(self.reportsFolder, 'FilingSummary.xml'))
        if self.summaryXslt and len(self.summaryXslt) > 0 :
            summary_transform = etree.XSLT(etree.parse(self.summaryXslt))
            result = summary_transform(rootETree, asPage=etree.XSLT.strparam('true'))
            IoManager.writeHtmlDoc(result, os.path.join(self.reportsFolder, 'FilingSummary.htm'))
        if self.auxMetadata: 
            summary.writeMetaFiles()
     
        if self.zipOutputFile:
            # The output must be zipped.
            zipdir = self.reportsFolder      
            self.zipOutputFile = join(zipdir, self.zipOutputFile)    
            if self.entrypoint == self.zipOutputFile:  # Check absolute path destinations
                #message = ErrorMgr.getError('INPUT_OUTPUT_SAME').format(self.zipOutputFile)
                self.logWarn("Input and output files are the same: {}".format(self.zipOutputFile))
            self.logDebug(_("Creating output {} containing rendering results and other input files."
                           ).format(self.zipOutputFile))
            try:
                zf = zipfile.ZipFile(self.zipOutputFile, 'w', allowZip64=False)                                             
                for f in os.listdir(self.reportsFolder):
                    if not isdir(f) and not IoManager.isFileHidden(f) and not f == basename(self.zipOutputFile):
                        IoManager.moveToZip(zf, join(zipdir, f), basename(f))
                # shutil.rmtree(self.reportsFolder)
            finally:
                zf.close()
            self.logDebug(_("Rendering results zip file {} populated").format(self.zipOutputFile))
            if self.isDaemon:
                try:
                    result = IoManager.move_clobbering_file(self.zipOutputFile, self.deliveryFolder) 
                    IoManager.move_clobbering_file(options.entrypoint, self.doneFile)
                    self.logDebug(_("Successfully post-processed to {}.").format(result))
                except OSError as err:
                    #self.logError(_(ErrorMgr.getError('POST_PROCESSING_ERROR').format(err)))
                    self.logError(_("Failure: Post-processing I/O or OS error: {}").format(err))

        if self.deleteProcessedFilings:
            for folder in self.createdFolders: shutil.rmtree(folder,ignore_errors=True) 
            
    
    def postprocessFailure(self, options):
        if self.isSingles:
            #message = ErrorMgr.getError('CANNOT_PROCESS_INPUT_FILE').format(self.entrypoint)
            self.logError("Cannot process input file {}.".format(self.entrypoint), file=__file__ + ' postprocessFailure')
        else:
            # In daemon mode, write an error log file looking somewhat like the one from RE2 and named the same.
            # Separately, create a zero-length "fail file" for the sole purpose of signaling status.
            if self.failFile is not None: 
                open(self.failFile, 'w').close()
            if self.entrypoint is None:
                #message = _(ErrorMgr.getError('CANNOT_PROCESS_INPUT_FILE')).format(self.entrypoint)
                self.logError(_("Cannot process input file {}.").format(self.entrypoint))
            else:
                errlogpath = join(self.deliveryFolder, os.path.splitext(self.zipOutputFile)[0] + '_errorLog.txt')
                if isfile(errlogpath): os.remove(errlogpath)
                #self.logError(_(ErrorMgr.getError('CANNOT_PROCESS_ZIP_FILE')).format(options.entrypoint))
                self.logError(_("Cannot process zip file {}; moving to fail folder.").format(options.entrypoint))
                IoManager.move_clobbering_file(options.entrypoint, self.errorsFolder)
                print(self.deliveryFolder + " " + errlogpath)
                with open(errlogpath, 'w', encoding='utf-8') as f:
                    for errmsg in self.ErrorMsgs:
                        message = "[" + errmsg.msgCode + "] " + errmsg.msg
                        print(message, file=f)
                    f.close() 


    
    def addToLog(self, message, messageArgs=(), messageCode='error', file=basename(__file__), level=logging.DEBUG):
        # Master log and error/warning msg handler            
        messageDict = {'fatal':logging.FATAL
                       , 'error':logging.ERROR
                       , 'warn':logging.WARN
                       , 'info':logging.INFO
                       , 'debug':logging.DEBUG
                       , 'trace':logging.NOTSET}
        # find a level that agrees with the code
        if messageCode not in messageDict: messageLevel = logging.CRITICAL
        else:  messageLevel = messageDict[messageCode.casefold()]
        # if both level and code were given, err on the side of more logging:
        messageLevel = max(level, messageLevel)
        if self.entrypoint is not None and len(self.instanceList + self.inlineList) > 1:
            message += ' --' + self.entrypoint
        message = message.encode('utf-8', 'replace').decode('utf-8')
        if messageLevel >= logging.INFO:
            self.ErrorMsgs.append(Errmsg(messageCode, message))

        if (self.modelManager and getattr(self.modelManager, 'modelXbrl', None)):
            self.modelManager.modelXbrl.logger.messageLevelFilter = None
            self.modelManager.modelXbrl.log(messageLevel, messageCode, message, *messageArgs)
        else:
            super().addToLog(message, messageArgs=messageArgs, messageCode=messageCode, file=file, level=messageLevel)
            
    # Lowercase tokens apparently write to standard output??
    
    def logTrace(self, message, messageArgs=(), file=basename(__file__)):
        self.addToLog(str(message), messageArgs=messageArgs, file=file, level=logging.NOTSET, messageCode='trace')

    def logDebug(self, message, messageArgs=(), file=basename(__file__)):
        self.addToLog(str(message), messageArgs=messageArgs, file=file, level=logging.DEBUG, messageCode='debug')

    def logInfo(self, message, messageArgs=(), file=None):
        self.addToLog(str(message), messageArgs=messageArgs, file=None, level=logging.INFO, messageCode='info')

    def logWarn(self, message, messageArgs=(), file=None):
        self.addToLog(str(message), messageArgs=messageArgs, file=None, level=logging.WARN, messageCode='warn')

    def logError(self, message, messageArgs=(), file=None):
        self.addToLog(str(message), messageArgs=messageArgs, file=None, level=logging.ERROR, messageCode='error')

    def logFatal(self, message, messageArgs=(), file=None):
        self.addToLog(str(message), messageArgs=messageArgs, file=None, level=logging.FATAL, messageCode='fatal')

    '''
    Errors and Logging
    '''
    
class Errmsg(object):
    def __init__(self, messageCode, message):
        self.msgCode = messageCode
        self.msg = message

if __name__ == "__main__":
    '''
    if '--COMserver' in sys.argv:
        from arelle import CntlrComServer
        CntlrComServer.main()
    else:
        main()
    '''
    main()

